from multiprocessing.connection import Client
import pigpio # http://abyz.co.uk/rpi/pigpio/python.html

class I2c_Lora_Module():

	def __init__(self):
		self.pi = pigpio.pi()
		if not self.pi.connected:
			print 'ERROR:pigpiod not running, start sudo pigpiod'
			exit()
		print 'PiGpiod running'

	def open(self, bus, addr):
		self.h = self.pi.i2c_open(bus, addr)
		# print 'Open ', bus, 'addr', addr
		if self.h >= 0:
			return True

		print 'I2c Module fail to open'
		return False

	def write(self, data):
		if self.h >= 0: # Connected OK?
			try:
				# print 'I2c Module write: ', str(data)
				self.pi.i2c_write_device(self.h, data)
				return True
			except pigpio.error as e:
				print (e)

		return False

	def read(self, bytes = 0):
		if self.h >= 0: # Connected OK?
			try:
				if bytes > 0:
					data = self.pi.i2c_read_device(self.h, bytes)
				else:
					data = self.pi.i2c_read_byte(self.h)
				# print 'I2c Module receive:', str(data)
				return data
			except pigpio.error as e:
				print ('I2c error:', e)
		return None

	def close(self):
		self.pi.i2c_close(self.h)


import os
import fnmatch
import time, random
import threading
import traceback
import struct
import sys
import binascii
import csv
import random

STANDARD = 0
FAVORITE = 1
REFERENCE = 2

MAX_MESSAGE_SIZE = 32

LORA_VERSION = 240

BUS = 1
SLAVE_ADDR=0x4C

#command codes to send to Lora MCU
WHOAMI = 0x55
PUSH_DATA = 0xBB
PULL_DATA = 0xCC
STATUS = 0xEE
CONFIGURE = 0xFF
RESET_DEV = 0xDD

#bitmasks for status messages
BUSY_FLG = 0
CONNECTED_FLG = 1
TX_READY_FLG = 2
CONFIGURED_FLG = 3
UNBLOCKED_FLG = 4
TX_DONE_FLG = 5
RX_DATA_FLG = 6

#uint8_t addr_enable;
#uint8_t data_rate;
#uint8_t public_network;
#uint8_t enable_duty_cycle;

ADR_ENABLE = 0x01

#between lowest of 0 (SF12) to max of 6 (SF7)
DATA_RATE = 0x00

#True for lorawan
PUBLIC_NETWORK = 0x01

# 1% duty cycle for EU868
ENABLE_DUTY_CYCLE = 0x00



#dev_eui,app_eui,app_key
config_array = bytearray([ 0xB1, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7, 0xB8,
							0x1C, 0x2C, 0x3C, 0x4C, 0x5C, 0x6C, 0x7C, 0x8C,
							0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6, 0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C,ADR_ENABLE,DATA_RATE,PUBLIC_NETWORK,ENABLE_DUTY_CYCLE])
configSize = 36

#config_array = bytearray()

class Lora():

	currentLoraDir = 'data/lora/'

	# Lora init function. All initializations should be made here
	def __init__(self, deviceId, mac):
		#Create file in server with format eg B2001.txt
		self.deviceId = deviceId
		self.mac_address = mac

		if not os.path.exists(self.currentLoraDir):
			original_umask = os.umask(0)
			try:
				os.makedirs(self.currentLoraDir, 0777)
			finally:
				os.umask(original_umask)

		self.numberLogsLora = self.numberLogsInLora()
		print('lora version: ', LORA_VERSION )
		print('Logs in lora: ' + str(self.numberLogsLora))

		# self.message_q = Queue.Queue()
		self.comm_thread = LoraCommunicate(self.currentLoraDir)

	#Number of tags to be sent
	def numberLogsInLora(self):
		if not os.path.exists(self.currentLoraDir):
			return 0

		n = len(fnmatch.filter(os.listdir(self.currentLoraDir),'*.csv'))
		return n

	# Write a file for Lora transmition
	def writeLineLora(self, tag, timeStr, id, type):

		#Create a file with the current time stamp
		fileName =  time.strftime("%Y%m%d%H%M%S")
		#Example to test multiple files
		#fileName =  "20161103010101"

		if os.path.exists(self.currentLoraDir + fileName + '.csv'):
			for x in range(0, 9999):
				fileNameNew = fileName + str(x).zfill(4)
				if not os.path.exists(self.currentLoraDir + fileNameNew + '.csv'):
					fileName = fileNameNew
					break

		fileName = self.currentLoraDir + fileName + '.csv'

		# Info formatation

		# tag, id, type are strings
		# timeStr is in format %Y-%m-%d %H:%M:%S

		dateHex = format(int(time.mktime(time.strptime(timeStr, '%Y-%m-%d %H:%M:%S'))) - time.timezone, 'x').upper()

		#use the id chars to store ref, if the tag is a reference
		if(type[0] == 'R' ):
			idChars = type.strip('Reference')
			row = (",".join([tag, dateHex, idChars, type[0]]))
		else:
			row = (",".join([tag, dateHex, id, type[0]]))

		# Write info to file
		with open(fileName, 'w') as f:
			f.write(row + '\n')
			f.close()
		os.chmod(fileName, 777)


	# Lora state machine that will run every loop
	def stateMachine(self):
		pass

	def setup(self):
		global config_array
		#dev_eui = bytearray([0,0,0,0,0,0,0,0])
		#app_eui = bytearray([0,0,0,0,0,0,0,0])

		#reset the device
		self.comm_thread.reset()
		time.sleep(1)
		app_key = bytearray([0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6, 0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C])

		#dev_eui = binascii.unhexlify(MAC_address.replace(':', ''))
		dev_eui = bytearray(binascii.unhexlify(self.mac_address.replace(b':', b'')))
		app_eui = bytearray(dev_eui)

		padding_mac = bytearray([0xFF,0xFF])
		if len(dev_eui) ==  6:
			dev_eui += padding_mac
			app_eui += padding_mac
		elif len(dev_eui) > 8:
			return False

		app_eui[2] ^= 0xFF
		app_eui[4] ^= 0xFF
		app_eui[6] ^= 0xFF

		#print 'Lora, current dev EUI: ', binascii.hexlify(dev_eui)
		#print 'Lora, current app EUI:', binascii.hexlify(app_eui)
		#print 'Lora, current app key', binascii.hexlify(app_key)
		config_array = dev_eui + app_eui + app_key
		config_array.append(ADR_ENABLE)
		config_array.append(DATA_RATE)
		config_array.append(PUBLIC_NETWORK)
		config_array.append(ENABLE_DUTY_CYCLE)
		#print 'Lora, current config array', binascii.hexlify(config_array)
		if (self.comm_thread.configureLora(config_array)):
			return True
		else:
			return False

	def runLoraThreads(self):
		self.comm_thread.start()
		print 'started Lora Threads'
		return

	def stopLoraThreads(self):
		self.monitor_thread.stop()
		self.comm_thread.stop()
		with self.message_q.mutex:
			self.message_q.queue.clear()
		print 'stopped Lora Threads, cleared Queue'
		return

	def reset(self):
		return self.comm_thread.reset()

	def isLoraPresent(self):
		return self.comm_thread.isLoraPresent()

	def isLoraConnected(self):
		return self.comm_thread.isLoraConnected()


###################################################################################
# NAME		: LoraMonitorLog
# DESCRIPTION : checks for new log entries, converts and populates the message queue
# INPUT	   : NONE
###################################################################################
class LoraMonitorLog():
	def __init__(self, monitor_dir):
		self.monitor_dir = os.path.abspath(monitor_dir)

	def file_is_empty(self, file_path):
		return os.stat(file_path).st_size == 0

	def deleteFile(self, file_path = None):
		if os.path.exists(file_path):
			os.remove(file_path)
			# print('deleted: ', file_path)
			return

		print('The file does not exist: ', file_path)
		return

	def processFile(self, file_path):
		tag_str = None

		try:
			# Check if file exist
			if not os.path.exists(file_path) or self.file_is_empty(file_path):
				return tag_str

			log_file = open(file_path, 'r')
			# print('opening and reading: ', file_path)

			line = log_file.readline()
			line = line.strip('\n')

			log_file.close()

			if line is not None and line != '':
				tag_str = self.convertToByteArray(line)

		except (IOError, Exception) as e:
			# Delete all files that are not correctly formated
			print('Process File', e)
			try:
				self.deleteFile(file_path)
			except:
				pass
			pass

		return tag_str

	#typedef struct BeamerPacket{
	#	uint8_t packetSize;
	#	uint8_t packetType;
	#	int32_t unixEpoch;
	#	uint8_t id_a;
	#	uint8_t id_b;
	#	uint8_t id_c;
	#	uint8_t cType;
	#	uint8_t data[MAX_TAG_SIZE];
	#} BeamerPacket;
	##################################################################
	def convertToByteArray(self,line):
		lines = line.rstrip()
		value_list = lines.split(",")
		#format tag
		tag = binascii.a2b_hex(value_list[0])
		#format epoch
		epoch_int = int(value_list[1],16)
		epoch = struct.pack("<I", epoch_int)
		#format ID chars
		id_a_char,id_b_char,id_c_char = value_list[2]
		id_a = struct.pack('<B', ord(id_a_char))
		id_b = struct.pack('<B', ord(id_b_char))
		id_c = struct.pack('<B', ord(id_c_char))
		#packet type byte
		packet_type_hex = 0x55
		packet_type = struct.pack('<B',packet_type_hex)
		#card type
		ctype_int = 0
		if value_list[3] == 'S':
			ctype_int = STANDARD
		elif value_list[3] == 'F':
			ctype_int = FAVORITE
		elif value_list[3] == 'R':
			ctype_int = REFERENCE
		ctype = struct.pack('<B',ctype_int)

		#includes size of itself
		length_int = 1 + len(packet_type) + len(epoch) + len(id_a) + len(id_b) + len(id_c) + len(ctype) + len(tag)
		length = struct.pack('<B', length_int)
		byte_str = length + packet_type + epoch + id_a + id_b + id_c + ctype + tag

		#print('byte_str:',binascii.hexlify(byte_str))
		byte_list = bytearray(byte_str)

		return byte_list

	def nextFile(self):

		file_path = None
		try:
			l = os.listdir(self.monitor_dir)
			for file in l:
				file_path = os.path.join(self.monitor_dir, file)
				if os.path.isfile(file_path):
					break

		except (FileNotFoundError, NotADirectoryError, IsADirectoryError):
			pass

		return file_path


###################################################################################
# NAME		: LoraCommunicate
# DESCRIPTION : Consumes the message queue from monitor thread, transmits via i2c
# INPUT	   : NONE
###################################################################################
class LoraCommunicate(threading.Thread):
	def __init__(self, lora_dir):
		threading.Thread.__init__(self)

		self.daemon = True
		self.setDaemon(True)

		self.i2c = I2c_Lora_Module()

		self.monitor = LoraMonitorLog(lora_dir)
		self.isConnected = False

	def testBit(self, int_type, offset):
		mask = 1 << offset
		if (int_type & mask):
			return True

		return False

	def BeamerWrite(self,command, size, dataToSend = [], *args):
		h = self.i2c.open(BUS, SLAVE_ADDR)
		sendBuffer = list(dataToSend)
		cmdBuffer = list([command,size, 0, 0 ,0, 0])

		status = 0
		#test
		if h: # Connected OK?
			success = False
			try:
				#print 'SIZE:', size
				#print('DATA: [%s]' % ', '.join(map(str,sendBuffer)))
				self.i2c.write(cmdBuffer)
				time.sleep(.2)
				self.i2c.write(sendBuffer)
				#Check status byte
				time.sleep(.2)
				status = self.i2c.read()
				#print 'status: ', status
				if status == 0 or status== None:
					success = False
					status = 0
					self.i2c.close()
					return status

				#print("PUSH OK")
				# if we get here, we succeeded, so break out of the loop
				success = True
			except e:
				print ('Write error', e)
				time.sleep(3)

			if not success:
				print ("*********************")
				print ("Beamer Write Failed!")
				print ("*********************")

		self.i2c.close()
		return status

	def BeamerRead(self,command, size, dataToRead = [], *args):
		h = self.i2c.open(BUS, SLAVE_ADDR)
		dataToReceive=[]
		cmdBuffer = list([command,size, 0, 0 ,0, 0])
		if h: # Connected OK?
			success = False
			try:
				#self.pi.i2c_write_byte(h, command)
				self.i2c.write(cmdBuffer)
				time.sleep(.1)
				(count, dataToReceive) = self.i2c.read(size)
				# if we get here, we succeeded, so break out of the loop
				success = True
				#print('[%s]' % ', '.join(map(str,dataToReceive)))
			except e:
				print ('Read error', e)
				time.sleep(3)

			if not success:
				print ("*********************")
				print ("Beamer Read Failed!")
				print ("*********************")
				dataToReceive = [0]

		self.i2c.close()
		return dataToReceive

	def configureLora(self, config = bytearray()):
		print('configuring LoRa')
		if (len(config) != configSize):
			print'Failed, configuration array != declared size'
			return False
		retval = self.BeamerWrite(CONFIGURE,configSize,config)
		if self.testBit(retval,CONFIGURED_FLG):
			return True
		else:
			return False

	def getLoraStatus(self):
		state = 0
		try:
			stat = self.BeamerRead(STATUS,1)
			state = stat[0]
		except:
			print 'could not get status, probably due to i2c read failure'

		return state

	def helpme(self):
		print 'help'

	def reset(self):
		try:
			self.BeamerRead(RESET_DEV, 1)
			return True
		except:
			print 'could not reset lora due to i2c failure'

		return False

	def isLoraPresent(self):
		try:
			stat = self.BeamerRead(STATUS,1)
			return self.testBit(stat[0],UNBLOCKED_FLG)

		except:
			print 'Could not get lora, setting not present'

		return False

	def isLoraConnected(self):
		if(self.isConnected):
			return True
		else:
			return False

	def sendMsgLora(self,msgToSend = []):
		#must be of 4 elements
		msgLength = len(msgToSend)
		#print 'msgLength: ', msgLength
		if (msgLength < 10) or (msgLength > MAX_MESSAGE_SIZE):
			print 'incorrect list size in sendMsgLora'
			return 0
		#pad zeroes on i2c message
		#justifyBytes = MAX_MESSAGE_SIZE - msgLength
		#print 'justifybites size', justifyBytes
		#zero_pad = bytearray(justifyBytes)
		#msgToSend += zero_pad
		#print 'raw message: ', binascii.hexlify(msgToSend)
		ret_status = self.BeamerWrite(PUSH_DATA,msgLength,msgToSend)
		return ret_status

	def LocalDeleteLoraLogFile(self,file_path = None):
		if os.path.exists(file_path):
			os.remove(file_path)
			print 'deleted: ', file_path
			return
		else:
			#print 'The file does not exist: ', file_path
			return

	def delay(self, t, r = None):
		if r is None:
			r = random.uniform(0, 3)
		time.sleep(t + r)
		return

	def loraWaitConnect(self):
		global config_array
		connection_attempts = 0
		while True:
			current_status = self.getLoraStatus()
			print 'Startup current status: ', current_status
			if not self.testBit(current_status,UNBLOCKED_FLG):
				print 'device is blocked, status: ', current_status
				self.isConnected = False
				# time.sleep(10)
				self.delay(5, 0)
				continue
			if not self.testBit(current_status,CONFIGURED_FLG):
				if (self.configureLora(config_array)):
					print 'Configured Successfully'
					self.isConnected = False
					# time.sleep(7)
					self.delay(5, 0)
				continue
			if not self.testBit(current_status,CONNECTED_FLG):
				print 'Configured, waiting to connect'
				self.isConnected = False
				connection_attempts += 1
				if(connection_attempts > 2):
					#print 'Too waiting too long to connect, resetting lora'
					#self.reset()
					# time.sleep(10)
					connection_attempts = 0
				# time.sleep(7)
				self.delay(10)
				continue
			if self.testBit(current_status,CONNECTED_FLG) and self.testBit(current_status,CONFIGURED_FLG):
				print 'Connected, ready to send'
				# time.sleep(3)
				self.delay(3)
				self.isConnected = True
				break

		return True

	def run(self):
		global config_array
		time.sleep(1)
		count=0
		tx_retry_count = 0
		lora_packet_tuple = tuple()
		local_lora_tuple = None


		monitor = LoraMonitorLog('data/lora/')

		while True:

			# Create a boot delay from 0 to 3 sec
			self.delay(0)
			self.loraWaitConnect()
			idle_state_change = True
			#normal operation begins here

			file = None

			while True:
				statusMsg = 0

				#tried to send same packet 3 times already. trigger watchdog
				if tx_retry_count > 3:
					print 'hit max retry count, resetting lora'
					# time.sleep(17)
					self.delay(17)
					break

 				file=monitor.nextFile()
				if local_lora_tuple is not None:
					idle_state_change = True
					print '-------------------------------------------------------'
					print 'resending local_lora_tuple'

				elif file is not None:
					idle_state_change = True

					lora_packet_tuple = monitor.processFile(file)

					if lora_packet_tuple is not None:
						local_lora_tuple = tuple(lora_packet_tuple)
						# print 'Sending: ', binascii.hexlify(local_lora_tuple)
						tx_retry_count = 0
				else:
					# time.sleep(3)
					self.delay(3, 0)
					current_status = self.getLoraStatus()
					if self.testBit(current_status,CONNECTED_FLG) and self.testBit(current_status,CONFIGURED_FLG) :
						if idle_state_change:
							print 'Idle status: ', current_status
							idle_state_change = False
						continue
					else:
						break

				if local_lora_tuple is not None:
					statusMsg = self.sendMsgLora(local_lora_tuple)
				print 'send msg status: ', statusMsg
				#break early, connection has been lost
				if not self.testBit(statusMsg,CONNECTED_FLG) and not self.testBit(statusMsg,CONFIGURED_FLG) :
					break
				#data has been scheduled for TX so we aren't TX_READY any more
				#if not self.testBit(statusMsg,TX_READY_FLG):
					#print 'TX_ready = 0: ', statusMsg

				tx_timeout = 0
				while(True):
					print 'packet sent to lora module,waiting for TX to complete'
					# time.sleep(3)
					self.delay(3, 0)
					waiting_tx_done = self.getLoraStatus()
					if self.testBit(waiting_tx_done,TX_DONE_FLG):
						count += 1
						print 'TX complete, sent packet number: ', count
						monitor.deleteFile(file)
						file = None

						local_lora_tuple = None
						# time.sleep(3)
						self.delay(3)
						break

					tx_timeout+=1
					if (tx_timeout>3):
						tx_retry_count += 1
						print 'TX timed  or failed out, retrying in 4 seconds'
						# time.sleep(4)
						self.delay(4)
						break

		self.i2c.stop()


# class I2c_Lora_Stub():
# 	def __init__(self):
# 		address = ('localhost', 6000)
# 		try:
# 			self.i2c = Client(address, authkey=b'i2c')
# 		except:
# 			print('Impossible to connect to i2c stub')
# 			exit(-1)
# 		return
# 	def open(self, bus, addr):
# 		self.i2c.send(['Start', bus, addr])
#
# 	def write(self, data):
# 		self.i2c.send([data])
# 		return True
#
# 	def read(self):
# 		try:
# 			msg = self.i2c.recv()
# 		except:
# 			return None
# 		return msg
#
# 	def close(self):
# 		self.i2c.send(['Close'])
# 		self.i2c.close()
