import os, fnmatch
from os import listdir
from os.path import isfile, join
import time
import threading
import Queue
import traceback
import struct
import sys
import binascii
import csv
import random
import pigpio # http://abyz.co.uk/rpi/pigpio/python.html
import pyinotify

STANDARD = 0
FAVORITE = 1
MAX_MESSAGE_SIZE = 32
Q_FILE_PATH = 0
Q_LORA_PACKET = 1

LORA_VERSION = 105

BUS = 1
SLAVE_ADDR=0x2B

#command codes to send to Lora MCU
WHOAMI = 0x55
PUSH_DATA = 0xBB
PULL_DATA = 0xCC
STATUS = 0xEE
CONFIGURE = 0xFF
RESET_DEV = 0xDD

#bitmasks for status messages
BUSY_FLG = 0
CONNECTED_FLG = 1
TX_READY_FLG = 2
CONFIGURED_FLG = 3
UNBLOCKED_FLG = 4
TX_DONE_FLG = 5
RX_DATA_FLG = 6

#dev_eui,app_eui,app_key
config_array = bytearray([ 0xB1, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7, 0xB8,
                            0x1C, 0x2C, 0x3C, 0x4C, 0x5C, 0x6C, 0x7C, 0x8C,
                            0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6, 0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C])
configSize = 32

#config_array = bytearray()

class Lora():

	currentLoraDir = 'data/lora/'



	# Lora init function. All initializations should be made here
	def __init__(self, deviceId):
		#Create file in server with format eg B2001.txt
		self.deviceId = deviceId

		if not os.path.exists(self.currentLoraDir):
			original_umask = os.umask(0)
			try:
				os.makedirs(self.currentLoraDir, 0777)
			finally:
				os.umask(original_umask)

		self.numberLogsLora = self.numberLogsInLora()
		print('lora version: ', LORA_VERSION )
		print('Logs in lora: ' + str(self.numberLogsLora))

		self.rpi=pigpio.pi()
		if not self.rpi.connected:
			print 'ERROR:pigpiod not running, start sudo pigpiod'
			exit()

		self.message_q = Queue.Queue()
		#self.message_q = multiprocessing.Queue()
		#self.message_q = collections.deque()
		self.monitor_thread = LoraMonitorLog(self.message_q,self.currentLoraDir)
		self.comm_thread = LoraCommunicate(self.message_q, self.rpi)



	#Number of tags to be sent
	def numberLogsInLora(self):
		if not os.path.exists(self.currentLoraDir):
			return 0
		else:
			n = len(fnmatch.filter(os.listdir(self.currentLoraDir),'*.csv'))
			return n

	# Write a file for Lora transmition
	def writeLineLora(self, tag, timeStr, id, type):

		#Create a file with the current time stamp
		fileName =  time.strftime("%Y%m%d%H%M%S")
		#Example to test multiple files
		#fileName =  "20161103010101"

		if os.path.exists(self.currentLoraDir + fileName + '.csv'):
			for x in range(0, 9999):
				fileNameNew = fileName + str(x).zfill(4)
				if not os.path.exists(self.currentLoraDir + fileNameNew + '.csv'):
					fileName = fileNameNew
					break

		fileName = self.currentLoraDir + fileName + '.csv'

		# Info formatation

		# tag, id, type are strings
		# timeStr is in format %Y-%m-%d %H:%M:%S

		dateHex = format(int(time.mktime(time.strptime(timeStr, '%Y-%m-%d %H:%M:%S'))) - time.timezone, 'x').upper()
		row = (",".join([tag, dateHex, id, type[0]]))

		# Write info to file
		with open(fileName, 'w') as f:
			f.write(row + '\n')
			f.close()
		os.chmod(fileName, 0o777)


	# Lora state machine that will run every loop
	def stateMachine(self):
		pass

	def setup(self, MAC_address):
		global config_array
		#dev_eui = bytearray([0,0,0,0,0,0,0,0])
		#app_eui = bytearray([0,0,0,0,0,0,0,0])
		app_key = bytearray([0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6, 0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C])

		#dev_eui = binascii.unhexlify(MAC_address.replace(':', ''))
		dev_eui = bytearray(binascii.unhexlify(MAC_address.replace(b':', b'')))
		app_eui = bytearray(dev_eui)

		padding_mac = bytearray([0xFF,0xFF])
		if len(dev_eui) ==  6:
			dev_eui += padding_mac
			app_eui += padding_mac
		elif len(dev_eui) > 8:
			return False

		app_eui[2] ^= 0xFF
		app_eui[4] ^= 0xFF
		app_eui[6] ^= 0xFF

		print 'Lora, current dev EUI: ', binascii.hexlify(dev_eui)
		print 'Lora, current app EUI:', binascii.hexlify(app_eui)
		print 'Lora, current app key', binascii.hexlify(app_key)
		config_array = dev_eui + app_eui + app_key
		print 'Lora, current config array', binascii.hexlify(config_array)
		if (self.comm_thread.configureLora(config_array)):
			return True
		else:
			return False

	def runLoraThreads(self):
		self.monitor_thread.start()
		self.comm_thread.start()
		print 'started Lora Threads'
		return

	def stopLoraThreads(self):
		self.monitor_thread.stop()
		self.comm_thread.stop()
		with self.message_q.mutex:
			self.message_q.queue.clear()
		print 'stopped Lora Threads, cleared Queue'
		return

	def reset(self):
		return self.comm_thread.reset()

	def isLoraPresent(self):
		return self.comm_thread.isLoraPresent()

	def isLoraConnected(self):
		return self.comm_thread.isLoraConnected()


###################################################################################
# NAME        : LoraMonitorLog
# DESCRIPTION : checks for new log entries, converts and populates the message queue
# INPUT       : NONE
###################################################################################
class LoraMonitorLog(threading.Thread):
	def __init__(self, out_q, monitor_dir):
		threading.Thread.__init__(self)
		self.out_q = out_q
		self.monitor_dir = monitor_dir
		self.daemon = True
		self.setDaemon(True)
		#self.LoraInstance = LoraCommunicate(self.out_q) 

	def process(self,line):
		print '>', line.strip('\n')

	def file_is_empty(self,file_path):
		return os.stat(file_path).st_size==0

	def deleteLoraLogFile(self,file_path = None):
		if os.path.exists(file_path):
			os.remove(file_path)
			print 'deleted: ', file_path
			return		
		else:
			print 'The file does not exist: ', file_path
			return

	def processFile(self, file_path):
		try:
			#------------------------- check
			if not os.path.exists(file_path):
				print 'process file path:', file_path
				print 'lora log file does not exist'
				time.sleep(1)
				return
			#------------------------- open
			loop_count = 0
			while self.file_is_empty(file_path):
				print 'file empty:', file_path	
				time.sleep(.5)
				loop_count+=1
				if loop_count > 4:
					print 'file empty, deleting', file_path
					self.deleteLoraLogFile(file_path)
					return

			log_file = open(file_path, 'r')
			print 'opening and reading: ', file_path
			log_file.seek(0,0)
			for line in log_file.readlines():
				self.process(line)

				if not line.strip():
					return
				else:
					#self.LoraInstance.sendLoraPacket(self.convertToByteArray(line))
					lora_data_tuple = (file_path,self.convertToByteArray(line))
					self.out_q.put(lora_data_tuple)
		except IOError:
			time.sleep(1)
		except:
			traceback.print_exc()
			time.sleep(1)


#typedef struct BeamerPacket{
#	uint8_t packetSize;
#	uint8_t packetType;
#	int32_t unixEpoch;
#	uint8_t id_a;
#	uint8_t id_b;
#	uint8_t id_c;
#	uint8_t cType;
#	uint8_t data[MAX_TAG_SIZE];
#} BeamerPacket;
##################################################################
	def convertToByteArray(self,line):
		lines = line.rstrip()
		value_list = lines.split(",")
		#format tag
		tag = value_list[0].decode('hex')
		#format epoch
		epoch_int = int(value_list[1],16)
		epoch = struct.pack("<I", epoch_int)
		#format ID chars
		id_a_char,id_b_char,id_c_char = value_list[2]
		id_a = struct.pack('<B', ord(id_a_char))
		id_b = struct.pack('<B', ord(id_b_char))
		id_c = struct.pack('<B', ord(id_c_char))
		#packet type byte
		packet_type_hex = 0x55
		packet_type = struct.pack('<B',packet_type_hex)
		#card type
		ctype_int = 0
		if value_list[3] == 'S':
			ctype_int = STANDARD
		elif value_list[3] == 'F':
			ctype_int = FAVORITE
		ctype = struct.pack('<B',ctype_int)

		#includes size of itself
		length_int = 1 + len(packet_type) + len(epoch) + len(id_a) + len(id_b) + len(id_c) + len(ctype) + len(tag)
		length = struct.pack('<B', length_int)
		byte_str = length + packet_type + epoch + id_a + id_b + id_c + ctype + tag

		#print 'byte_str',binascii.hexlify(byte_str)
		#print 'length(hex) ', binascii.hexlify(length)
		#print 'length(int) ', length_int
		byte_list = bytearray(byte_str)

		return byte_list

	def run(self):
		try:
			existing_lora_files = [f for f in listdir( os.path.abspath(self.monitor_dir)) if isfile(join( os.path.abspath(self.monitor_dir), f))]

			for file_itr in existing_lora_files:
				file_abs_path = os.path.abspath(self.monitor_dir)+ '/' + file_itr
				self.processFile(file_abs_path)

			# watch manager
			wm = pyinotify.WatchManager()
			wm.add_watch(os.path.abspath(self.monitor_dir), pyinotify.IN_CREATE, rec=True)
			print'Started Watch manager at: ',self.monitor_dir
			# event handler
			eh = LoraLogEventHandler(self)
			#self.LoraInstance.loraWaitConnect()
			# notifier
			notifier = pyinotify.Notifier(wm, eh)
			notifier.loop()

		except (KeyboardInterrupt, SystemExit):
			return


###################################################################################
# NAME        : LoraLogEventHandler
# DESCRIPTION : Fired as soon as a new file appears in the folder,
# INPUT       : NONE
###################################################################################
class LoraLogEventHandler(pyinotify.ProcessEvent):
	def __init__(self, monitor_instance):
		self.monitor_instance = monitor_instance

	def process_IN_CREATE(self, event):
		#print "CREATE event:", event.pathname
		self.monitor_instance.processFile(event.pathname)

	#not in use, add the IN_DELETE flag to the watch manager
	def process_IN_DELETE(self, event):
		print "DELETE event:", event.pathname


###################################################################################
# NAME        : LoraCommunicate
# DESCRIPTION : Consumes the message queue from monitor thread, transmits via i2c
# INPUT       : NONE
###################################################################################
class LoraCommunicate(threading.Thread):
	def __init__(self, in_q, pigpio_instance=pigpio.pi()):
		threading.Thread.__init__(self)
		self.in_q = in_q
		self.daemon = True
		self.setDaemon(True)
		self.pi = pigpio_instance
		self.isConnected = False

	def testBit(self, int_type, offset):
		mask = 1 << offset
		if (int_type & mask):
			return True

		return False

	def BeamerWrite(self,command, size, dataToSend = [], *args):
		h = self.pi.i2c_open(BUS, SLAVE_ADDR)
		sendBuffer = list(dataToSend)
		#sendBuffer.insert(0,size)
		sendBuffer.insert(0,command)
		status = 0
		#test
		if h >= 0: # Connected OK?
			success = False
			try:
				#print('[%s]' % ', '.join(map(str,sendBuffer)))
				#self.pi.i2c_write_byte(h,command)
				self.pi.i2c_write_device(h,sendBuffer)
				#Check status byte
				time.sleep(.2)
				status = self.pi.i2c_read_byte(h)
				print 'status: ', status
				if (status == 0):
						success = False
						return status

				#print("PUSH OK")
				# if we get here, we succeeded, so break out of the loop
				success = True
			except pigpio.error as e:
				print (e)
				time.sleep(3)
			if not success:
				print ("Failed to Write!")
		self.pi.i2c_close(h)
		return status

	def BeamerRead(self,command, size, dataToRead = [], *args):
		h = self.pi.i2c_open(BUS, SLAVE_ADDR)

		dataToReceive=[]

		if h >= 0: # Connected OK?
			success = False
			try:
				self.pi.i2c_write_byte(h, command)
				time.sleep(.1)
				(count, dataToReceive) = self.pi.i2c_read_device(h,size)
				# if we get here, we succeeded, so break out of the loop
				success = True
				#print('[%s]' % ', '.join(map(str,dataToReceive)))
			except pigpio.error as e:
				print (e)
				time.sleep(3)
			if not success:
				print ("Failed to Read!")
				dataToReceive = [0]
		self.pi.i2c_close(h)
		return dataToReceive

	def configureLora(self, config = bytearray()):
		print('configuring LoRa')
		if (len(config) != configSize):
			print'Failed, configuration array != declared size'
			return False
		retval = self.BeamerWrite(CONFIGURE,configSize,config)
		if self.testBit(retval,CONFIGURED_FLG):
			return True
		else:
			return False

	def getLoraStatus(self):
		state = 0
		try:
			stat = self.BeamerRead(STATUS,1)
			state = stat[0]
		except:
			print 'could not get status, probably due to i2c read failure'

		return state

	def helpme(self):
		print 'help'

	def reset(self):
		try:
			self.BeamerRead(RESET_DEV, 1)
			return True
		except:
			print 'could not reset lora due to i2c failure'

		return False

	def isLoraPresent(self):
		state = 0
		try:
			stat = self.BeamerRead(STATUS,1)
			return self.testBit(stat[0],UNBLOCKED_FLG)

		except:
			print 'Could not get lora, setting not present'

		return False

	def isLoraConnected(self):
		if(self.isConnected):
			return True
		else:
			return False

	def sendMsgLora(self,msgToSend = []):
		#must be of 4 elements
		msgLength = len(msgToSend)
		print 'msgLength: ', msgLength
		if (msgLength < 10) or (msgLength > MAX_MESSAGE_SIZE):
			print 'incorrect list size in sendMsgLora'
			return 0
		#pad zeroes on i2c message
		justifyBytes = MAX_MESSAGE_SIZE - msgLength
		#print 'justifybites size', justifyBytes
		zero_pad = bytearray(justifyBytes)
		msgToSend += zero_pad
		#print 'raw message: ', binascii.hexlify(msgToSend)
		ret_status = self.BeamerWrite(PUSH_DATA,msgLength,msgToSend)
		return ret_status
	
	def LocalDeleteLoraLogFile(self,file_path = None):
		if os.path.exists(file_path):
			os.remove(file_path)
			print 'deleted: ', file_path
			return		
		else:
			print 'The file does not exist: ', file_path
			return

	def loraWaitConnect(self):
		connection_attempts = 0
		while True:
			current_status = self.getLoraStatus()
			print 'Startup current status: ', current_status
			if not self.testBit(current_status,UNBLOCKED_FLG):
				print 'device is blocked, status: ', current_status
				self.isConnected = False
				time.sleep(10)
				continue
			if not self.testBit(current_status,CONFIGURED_FLG): 
				if (self.configureLora(config_array)):
					print 'Configured Successfully'
					self.isConnected = False
					time.sleep(5)
				continue
			if not self.testBit(current_status,CONNECTED_FLG):
				print 'Configured, waiting to connect'
				self.isConnected = False
				connection_attempts += 1
				if(connection_attempts > 12):
					self.reset()
				time.sleep(10)
				continue
			if self.testBit(current_status,CONNECTED_FLG) and self.testBit(current_status,CONFIGURED_FLG):
				print 'Connected, ready to send'
				time.sleep(3)
				self.isConnected = True
				break
		
		return True
		
	def sendLoraPacket(self, msgToSend=[]):
		dummy_msg = bytearray(32)
		print' '
		print 'MSG TO SEND: ', binascii.hexlify(msgToSend)
		current_status = self.getLoraStatus()
		if not self.testBit(current_status,TX_READY_FLG):
			print 'TX_READY = 0'
			return False
		else:
			print 'TX_READY = 1'

		dummy_statusMsg = self.sendMsgLora(msgToSend)
		if not self.testBit(dummy_statusMsg,TX_READY_FLG):
			print 'Did not Receive dummy I2C packet, TX_READY = 1'

		#time.sleep(2)

		statusMsg = self.sendMsgLora(msgToSend)
		if not self.testBit(statusMsg,TX_READY_FLG):
			print 'Did not Receive I2C packet, TX_READY = 1'
			return False

		time.sleep(5)

		is_tx_done = self.getLoraStatus()
		if self.testBit(is_tx_done,TX_DONE_FLG):
			print 'packet sent successfully, TX_DONE = 1'
			return True
		else:
			print 'packet failed, TX_DONE = 0'
			return False
		
	# def run(self):
	# 	while True:
	# 		if not self.in_q.empty():
	# 			print '-------------------------------------------------------'
	# 			print 'in_q.get(), Queue Depth ', self.in_q.qsize()
	# 			lora_packet_tuple = self.in_q.get()
	# 			temp_tuple = tuple(lora_packet_tuple)
	# 			print 'in_q.get(): ', binascii.hexlify(temp_tuple[Q_LORA_PACKET])
	# 			self.in_q.task_done()




	def run(self):
		global config_array
		time.sleep(1)   
		count=0
		
		delay = 1
		local_q = Queue.LifoQueue()
		lora_packet_tuple = tuple()
		local_lora_tuple = tuple()
		while True:

			self.loraWaitConnect()
			
			#normal operation begins here
			while True:
				statusMsg = 0
				
				if not local_q.empty():
					print 'local_q.get(), Queue Depth ', local_q.qsize()
					local_lora_tuple = local_q.get_nowait()
					if not (local_lora_tuple == None):
						self.in_q.put(local_lora_tuple)
						local_q.task_done()
				
				if not self.in_q.empty():
					print '-------------------------------------------------------'
					print 'in_q.get(), Queue Depth ', self.in_q.qsize()
					lora_packet_tuple = self.in_q.get_nowait()			
					temp_tuple = tuple(lora_packet_tuple)
					print 'in_q.get(): ', binascii.hexlify(temp_tuple[Q_LORA_PACKET])
					if not (lora_packet_tuple == None):
						local_q.put(lora_packet_tuple)
				
				else:
					time.sleep(3)
					current_status = self.getLoraStatus()
					if self.testBit(current_status,CONNECTED_FLG) and self.testBit(current_status,CONFIGURED_FLG) :
						print 'Idle status: ', current_status
						continue
					else:
						break

				dummyStatusMsg = self.sendMsgLora(lora_packet_tuple[Q_LORA_PACKET])
				if not self.testBit(dummyStatusMsg,TX_READY_FLG):
					print 'DUMMY TX_READY = 0'
					time.sleep(3)
				else:
					print 'DUMMY TX_READY = 1'
					time.sleep(1)
				statusMsg = self.sendMsgLora(lora_packet_tuple[Q_LORA_PACKET])
				print 'send msg status: ', statusMsg
				if not self.testBit(statusMsg,UNBLOCKED_FLG):
					break
				if not self.testBit(statusMsg,CONNECTED_FLG):
					break
				#data has been scheduled for TX so we aren't TX_READY any more
				if not self.testBit(statusMsg,TX_READY_FLG):
					tx_timeout = 0
					while(True):
						print 'packet sent to lora module,waiting for TX to complete'

						time.sleep(2 + delay)
						waiting_tx_done = self.getLoraStatus()
						if self.testBit(waiting_tx_done,TX_DONE_FLG):
							count += 1
							print 'TX complete, sent packet number: ', count
							print 'current delay: ', delay
							dispose = local_q.get_nowait()
							if not (dispose == None):
								local_q.task_done()
								print 'Queue task done'
							self.LocalDeleteLoraLogFile(lora_packet_tuple[Q_FILE_PATH])
							delay -= 1
							if (delay < 1):
								delay = 1
							time.sleep(2 + delay)
							break

						tx_timeout+=1
						if (tx_timeout>3):
							delay+=1
							if delay > 20:
								delay = 20
								print 'delay hit maximum: ', delay

							print 'TX timed out, retrying in 10 seconds'
							time.sleep(10)
							break
				else:
					#return to outer loop
					print 'TX not ready, retrying soon'
					time.sleep(3 + delay)
					break
		self.pi.stop()
##################################################################

#def main():
#	print 'starting thread'
#	q = Queue.Queue()
#
#	rpi=pigpio.pi()
#	if not rpi.connected:
#		print 'ERROR:pigpiod not running, start sudo pigpiod'
#		exit()
#
#	MonitorLog = MonitorLoraLogThread(q)
#	MonitorLog.start()
#	TransmitPacket = BeamerLoraThread(q, rpi)
#	TransmitPacket.start()
#	counter = 0
#	while True:
#		time.sleep(1)
#		counter += 1
#		print 'counter ', counter
#		#if not q.empty():
#		#	loglist_tuple = q.get()
#		#	print 'consuming queue: ', binascii.hexlify(loglist_tuple[Q_LORA_PACKET])
#		#	print 'from file: ', loglist_tuple[Q_FILE_PATH]
#		#	q.task_done()

#if __name__ == '__main__':
#        main()
