import nfc
import ndef
from nfc.tag import tt1
from nfc.tag import tt2
from nfc.tag import tt3
from nfc.tag import tt4

import array
import binascii
from operator import xor
import csv
import time
from datetime import datetime

FAVORITE_FILE = "data/favorite-cards.csv"

###################################################################################
#                TAG
###################################################################################


tagtypes = (
    ('uid', nfc.tag.tt1.Type1Tag),
    ('uid', nfc.tag.tt2.Type2Tag),
    ('idm', nfc.tag.tt3.Type3Tag),
    ('uid', nfc.tag.tt4.Type4Tag)
)


class TagsReader():

	TAG_READ_ERROR      = None
	TAG_ERROR           = -1
	TAG_OK              = 0
	TAG_OK_ADD          = 1
	TAG_FAVORITE        = 2
	TAG_SHUTDOWN        = 3
	TAG_UPDATE          = 4
	TAG_UPLOAD          = 5
	TAG_CHANGE_ID_NAME  = 6
	TAG_BEAMIAN_ID      = 7
	TAG_WIFI_CONNECT    = 8
	TAG_WIFI_DISCONNECT = 9

	MSG_TAG_OK              = ''
	MSG_TAG_FAVORITE        = 'Favorite'
	MSG_TAG_SHUTDOWN        = 'Turning Off'
	MSG_TAG_UPDATE          = 'Updating Firmware'
	MSG_TAG_UPLOAD          = 'Uploading Log File'
	MSG_TAG_WIFI_CONNECT    = 'Connecting to a WiFi Network'
	MSG_TAG_WIFI_DISCONNECT = 'Disconnecting WiFi'
	MSG_TAG_CRC_ERROR       = 'NTAG CRC not valid'
	MSG_TAG_INVALID_PAYLOAD = 'Invalid payload'

	###################################################################################
	# NAME        : TagsReader
	# DESCRIPTION : Deals with all TAG operations. Only works with 4 and 7 bytes.
	#             : Payload only on NTAG
	###################################################################################
	def __init__(self, deviceId, logFunction, logFunctionLora):

		self.clf = nfc.ContactlessFrontend('tty:S0:pn532')
		self.deviceId = deviceId
		self.logFunction = logFunction
		self.logFunctionLora = logFunctionLora
		self.lastTag = ''

		#Favorite Cards
		self.favorites = []
		self.setFavorite(False)

		try:
			#import tags list
			with open(FAVORITE_FILE, 'r') as csvfile:
				lines = csv.reader(csvfile, delimiter=',')
				lines = filter(None, lines)
				for l in lines:
					if l != None and l[1] != None and l[1] != '':
						self.favorites.append(l[1].lower())

			self.favorites.sort()
		except Exception, e:
			print('Error loading favorite csv: ' + str(e))

		#print(self.favorites)
        #why this function!??!?
	def connected(self, tag):

		print('connected')
		print(tag.type)
		for uid, type in tagtypes:
			if isinstance(tag, type):
				print(str(attr(tag, uid)).encode("hex"))
				return
			print("error: unknown tag type")
	###################################################################################
	# NAME        : wait
	# DESCRIPTION : Wait for a new tag
	# OUTPUT      : TAG UID
	###################################################################################
	def wait(self):

		rdwr_options = {
			'on-connect': lambda tag: False,
            'interval': 0.1
		}
		tag = self.clf.connect(rdwr=rdwr_options)

		return tag

	###################################################################################
	# NAME        : close
	# DESCRIPTION : Close connection with TAG reader
	###################################################################################
	def close(self):
		self.clf.close()

	###################################################################################
	# NAME        : waitAndSort
	# DESCRIPTION : Try read the UID tag.
	#             : If it is ntag try to read the content (Payload)
	###################################################################################
	def waitAndSort(self):

		try:
			uid = None

			# only return with a valid tag
			tag = self.wait()
			print(str(tag.identifier).encode('hex'))
			print(tag.type)

			if tag.ndef:
				for record in tag.ndef.records:
					print(record)
					print('\n')



			tagId=str(tag.identifier).encode('hex')


			if( tag.ndef is not None and tag.ndef.length > 0):

				#only read the 1st record (0)
				l = tag.ndef.records
				if( len(l) > 0):
					#print(l[0].type)
					#print(l[0].name)
					#print(l[0].data.encode('hex'))
					#header = l[0].data[:3] # "\0x02, 'e', 'n' " code UTF-8

					#compare only the text
					ndefText = l[0].data[3:]
					(r, m) = self.decodePayload(ndefText)
					if(r != None):
						return r, m

			return self.logTag(tagId)

		except Exception, e:
			print str(e)
			return self.TAG_READ_ERROR, self.MSG_TAG_OK

	###################################################################################
	# NAME        : decodePayload
	# DESCRIPTION : Decodes the payload in NTAG
	###################################################################################
	def decodePayload(self, s):

		print('decodePayload: ' + str(s))
		if s.lower() == "turnoff" or  s.lower() == "shutdown":
			return self.TAG_SHUTDOWN, self.MSG_TAG_SHUTDOWN

		if s.lower() == "update":
			return self.TAG_UPDATE, self.MSG_TAG_UPDATE

		if s.lower() == "upload" or s.lower() == "send":
			return self.TAG_UPLOAD, self.MSG_TAG_UPLOAD

		if s.lower()[:11] == "beamianidb2":
			return self.TAG_BEAMIAN_ID, s[9:].upper()

		if s.lower() == "favorite":
			return self.toggleFavorite()

		if s.lower()[:2] == "id":
			return self.TAG_CHANGE_ID_NAME, s[2:]

		if s.lower() == "connect":
			return self.TAG_WIFI_CONNECT, self.MSG_TAG_WIFI_CONNECT

		if s.lower() == "disconnect":
			return self.TAG_WIFI_DISCONNECT, self.MSG_TAG_WIFI_DISCONNECT

		print 'Unknown payload'

		return self.TAG_READ_ERROR, self.MSG_TAG_INVALID_PAYLOAD


	###################################################################################
	# NAME        : setFavorite
	# DESCRIPTION : Set the Favorite status
	###################################################################################
	def setFavorite(self, state):
		self.waitingFavorite = state

	###################################################################################
	# NAME        : toggleFavorite
	# DESCRIPTION : Toggle favorite status
	###################################################################################
	def toggleFavorite(self):

		if self.waitingFavorite:
			self.waitingFavorite = False
			return self.TAG_OK, self.MSG_TAG_OK
		else:
			self.waitingFavorite = True
			return self.TAG_FAVORITE, self.MSG_TAG_FAVORITE

	###################################################################################
	# NAME        : isWaitingFavorite
	# DESCRIPTION : The system is wainting or not for a new tag to save as favorite
	# OUTPUT      : True  - Is waiting for another tag
	#             : False -
	###################################################################################
	def isWaitingFavorite(self):
		return self.waitingFavorite

	###################################################################################
	# NAME        : logTag
	# DESCRIPTION : Save the tag ID and informations on log
	# INPUT       : tagId - tag uid
	###################################################################################
	def logTag(self, tagId):

		if tagId == None or tagId == '':
			return self.TAG_READ_ERROR, self.MSG_TAG_OK

		timeStr = datetime.now().strftime('%Y-%m-%d %H:%M:%S')

		if tagId in self.favorites:
			return self.toggleFavorite()
		else:
			if self.isWaitingFavorite():
				cardType = 'Favorite'
				self.setFavorite(False)
			else:
				cardType = 'Standard'
				#avoid duplicate last tag
				if self.lastTag == tagId:
					return self.TAG_OK, self.MSG_TAG_OK

		#save in csv format
		self.logFunction(tagId.upper(), timeStr, self.deviceId, cardType)
		self.logFunctionLora(tagId.upper(), timeStr, self.deviceId, cardType)
		self.lastTag = tagId

		print("Tag saved: " + str(tagId))

		return self.TAG_OK_ADD, self.MSG_TAG_OK
