from pirc522 import RFID
import array
import binascii
from operator import xor
import csv
from datetime import datetime
# from live_interaction import LiveInteraction

# live_interaction = LiveInteraction()
		
FAVORITE_FILE = "data/favorite-cards.csv"

###################################################################################
#                TAG
###################################################################################

class TagsReader():
	
	TAG_READ_ERROR      = None
	TAG_ERROR           = -1
	TAG_OK              = 0
	TAG_OK_ADD          = 1
	TAG_FAVORITE        = 2	
	TAG_SHUTDOWN        = 3
	TAG_UPDATE          = 4
	TAG_UPLOAD          = 5
	TAG_CHANGE_ID_NAME  = 6
	TAG_BEAMIAN_ID      = 7
	TAG_WIFI_CONNECT    = 8
	TAG_WIFI_DISCONNECT = 9
	TAG_REFERENCE       = 10
	TAG_KISS            = 11
	TAG_STATUS          = 12
	TAG_AUTO_TEST_START = 13
	TAG_AUTO_TEST_STOP  = 14
	TAG_CLEAR_ALL_TAGS  = 15
	TAG_CLEAR_WIFI_TAGS = 16
	TAG_CLEAR_LORA_TAGS = 17

	MSG_TAG_OK              = ''
	MSG_TAG_FAVORITE        = 'Favorite'
	MSG_TAG_SHUTDOWN        = 'Turning Off'
	MSG_TAG_UPDATE          = 'Updating Firmware'
	MSG_TAG_UPLOAD          = 'Uploading Log File'
	MSG_TAG_WIFI_CONNECT    = 'Connecting to a WiFi Network'
	MSG_TAG_WIFI_DISCONNECT = 'Disconnecting WiFi'
	MSG_TAG_CRC_ERROR       = 'NTAG CRC not valid'
	MSG_TAG_INVALID_PAYLOAD = 'Invalid payload'
	MSG_TAG_REFERENCE       = 'Reference '
	MSG_TAG_AUTO_TEST_START = 'Start automatic test'
	MSG_TAG_AUTO_TEST_STOP  = 'Stop automatic test'
	MSG_CLEAR_ALL_TAGS	    = 'Clear ALL tags'
	MSG_CLEAR_WIFI_TAGS	    = 'Clear WIFI tags'
	MSG_CLEAR_LORA_TAGS	    = 'Clear LORA tags'
	MSG_TAG_KISS            = 'Uploading last interaction'
	MSG_TAG_STATUS          = 'Version 1.12'
	
	###################################################################################
	# NAME        : TagsReader
	# DESCRIPTION : Deals with all TAG operations. Only works with 4 and 7 bytes.
	#             : Payload only on NTAG
	################################################################################### 
	def __init__(self, deviceId, logFunction):
		#                bus, dev,   speed, rst, ce, IOR
		self.rfid = RFID(  0,   0, 1000000,  15,  0,  29)
		self.util = self.rfid.util()
		self.util.debug = True
		self.deviceId = deviceId
		self.logFunction = logFunction
		self.lastTag = ''
		#self.rfid.set_antenna_gain(3)
		
		#Favorite Cards
		self.favorites = []
		self.setFavorite(False)
		#Reference
		self.setReference(False, None)
		
		try:
			#import tags list
			with open(FAVORITE_FILE, 'r') as csvfile:
				lines = csv.reader(csvfile, delimiter=',')
				lines = filter(None, lines)
				for l in lines:
					if l != None and l[1] != None and l[1] != '':
						self.favorites.append(l[1].lower())
			
			self.favorites.sort()
		except Exception, e:
			print('Error loading favorite csv: ' + str(e))
			
		#print(self.favorites)
	
	
	###################################################################################
	# NAME        : card_write
	# DESCRIPTION : Write a Command and Data to TAG213
	#             : Copy from library. Test with other reg values
	################################################################################### 
	def card_write(self, command, data):
		back_data = []
		back_length = 0
		error = False
		irq = 0x00
		irq_wait = 0x00
		last_bits = None
		n = 0

		if command == self.rfid.mode_auth:
			irq = 0x12
			irq_wait = 0x10
		if command == self.rfid.mode_transrec:
			irq = 0x77
			irq_wait = 0x30

		self.rfid.dev_write(0x02, irq | 0x80)
		self.rfid.clear_bitmask(0x04, 0x80)
		self.rfid.set_bitmask(0x0A, 0x80)
		self.rfid.dev_write(0x01, self.rfid.mode_idle)

		for i in range(len(data)):
			self.rfid.dev_write(0x09, data[i])

		self.rfid.dev_write(0x01, command)

		if command == self.rfid.mode_transrec:
			self.rfid.set_bitmask(0x0D, 0x80)

		i = 2000
		while True:
			n = self.rfid.dev_read(0x04)
			i -= 1
			if ~((i != 0) and ~(n & 0x01) and ~(n & irq_wait)):
				break

		self.rfid.clear_bitmask(0x0D, 0x80)

		if i != 0:
			if (self.rfid.dev_read(0x06) & 0x1B) == 0x00:
				error = False

				if n & irq & 0x01:
					print("E1")
					error = True

				if command == self.rfid.mode_transrec:
					n = self.rfid.dev_read(0x0A)
					last_bits = self.rfid.dev_read(0x0C) & 0x07
					if last_bits != 0:
						back_length = (n - 1) * 8 + last_bits
					else:
						back_length = n * 8

					if n == 0:
						n = 1

					if n > self.rfid.length:
						n = self.rfid.length

					for i in range(n):
						back_data.append(self.rfid.dev_read(0x09))
			else:
				print("E2")
				error = True

		return (error, back_data, back_length)

	###################################################################################
	# NAME        : anticoll
	# DESCRIPTION : Read the tag UID
	#             : Copy from library. Try multi IDs (4, 7 and 11 bytes)
	################################################################################### 
	def anticoll(self):
		
		back_data = []
		serial_number = []

		serial_number_check = 0

		self.rfid.dev_write(0x0D, 0x00)
		
		serial_number.append(0x93)
		serial_number.append(0x20)

		(error, back_data, back_bits) = self.card_write(self.rfid.mode_transrec, serial_number)
		
		if not error:
			if len(back_data) == 5:
				for i in range(4):
					serial_number_check = serial_number_check ^ back_data[i]
						
				if serial_number_check != back_data[4]:
					error = True
			
			else:
				error = True

		return (error, back_data)

	###################################################################################
	# NAME        : wait
	# DESCRIPTION : Wait for a new tag
	# OUTPUT      : TAG UID
	################################################################################### 
	def wait(self):
		self.rfid.wait_for_tag()
		(error, data) = self.rfid.request()
		print(data)
		if not error:
			(error, uid) = self.anticoll()
			if not error:
				print('[wait] UID: ' + str(uid))					
				return uid
				
		return None

	###################################################################################
	# NAME        : close
	# DESCRIPTION : Close connection with TAG reader
	################################################################################### 		
	def close(self):
		self.rfid.cleanup()

	###################################################################################
	# NAME        : waitAndSort
	# DESCRIPTION : Try read the UID tag.
	#             : If it is ntag try to read the content (Payload)
	################################################################################### 
	def waitAndSort(self):
		
		try:
			uid = None

			#wait for new tag
			uid = None
			while uid == None:
				uid = self.wait()
			
			if uid[0] != 0x88:
				#print('No Ntag')
				return self.logTag(binascii.hexlify(bytearray(uid[0:4])).lower()) 
			
			
			e0=True
			e1=True
			e2=True
			e3=True
			
			(e0, d0) = self.rfid.read(0)
			(e1, d1) = self.rfid.read(4)
			(e2, d2) = self.rfid.read(8)
			(e3, d3) = self.rfid.read(12)
	
			#needs to read the payload
			if (e0 or e1 or e2 or e3) or (d0 == None or d1 == None or d2 == None or d3 == None  ):
			#if (e0 or e1 or e2) or (d0 == None or d1 == None or d2 == None ):
				return self.TAG_READ_ERROR, self.MSG_TAG_OK

			data = []
			data.extend(d0)
			data.extend(d1)
			data.extend(d2)
			data.extend(d3)
			
			#parsing
			tagId  = data[0:3]; tagId.extend(data[4:8])
			tagId  = binascii.hexlify(bytearray(tagId)).lower()
			
			#print(tagId)
			#fazer operacoes com o tag ID para confirmar
			
			if xor(xor(xor(data[0], data[1]), data[2]), 0x88) != data[3] or xor(xor(xor(data[4], data[5]), data[6]), data[7]) != data[8]:
				#Not a NTAG
				#return self.logTag(binascii.hexlify(bytearray(uid[0:4])).lower())
				return self.TAG_ERROR, self.MSG_TAG_CRC_ERROR
			
			#TAG215 TAG216		
			if data[16] == 0x03:
				nBytes = data[17]
				tagPayload = data[18:18+nBytes]
				if (18+nBytes)>=64 or data[18+nBytes] != 0xfe:
					return self.logTag(tagId)
			
			#TAG213
			else:
				nBytes = data[22]
				tagPayload = data[23:23+nBytes]
				if (23+nBytes)>=64 or data[23+nBytes] != 0xfe:
					return self.logTag(tagId)
						

			#is a blank TAG?
			if nBytes == 0:
				return self.logTag(tagId)
			
			#Basic UTF8?
			if tagPayload[0] == 0xd1 and tagPayload[1] == 0x01 and tagPayload[3] == 0x54:
				utfPayload = tagPayload[7:nBytes]
				utfPayload = array.array('b', utfPayload).tostring().decode('utf-8')
				(r, m) = self.decodePayload(utfPayload)
				if(r == None):
					return self.logTag(tagId)
				return r, m
			
			print('Invalid payload')
			return self.TAG_ERROR, self.MSG_TAG_INVALID_PAYLOAD

		except Exception, e:
			print str(e)
			return self.TAG_READ_ERROR, self.MSG_TAG_OK
		
	###################################################################################
	# NAME        : decodePayload
	# DESCRIPTION : Decodes the payload in NTAG
	################################################################################### 
	def decodePayload(self, s):

		print('decodePayload: ' + str(s))
		if s.lower() == "turnoff" or  s.lower() == "shutdown":
			return self.TAG_SHUTDOWN, self.MSG_TAG_SHUTDOWN
		
		if s.lower() == "update":
			return self.TAG_UPDATE, self.MSG_TAG_UPDATE

		if s.lower() == "upload" or s.lower() == "send":
			return self.TAG_UPLOAD, self.MSG_TAG_UPLOAD
		
		if s.lower()[:11] == "beamianidb2":
			return self.TAG_BEAMIAN_ID, s[9:].upper()
		
		if s.lower() == "favorite":
			return self.toggleFavorite()

		if s.lower()[:2] == "id":
			return self.TAG_CHANGE_ID_NAME, s[2:]

		if s.lower()[:3] == "ref":
			return self.toggleReference(s[3:])

		if s.lower() == "connect":
			return self.TAG_WIFI_CONNECT, self.MSG_TAG_WIFI_CONNECT

		if s.lower() == "disconnect":
			return self.TAG_WIFI_DISCONNECT, self.MSG_TAG_WIFI_DISCONNECT

		if s.lower() == "starttest":
			return self.TAG_AUTO_TEST_START, self.MSG_TAG_AUTO_TEST_START

		if s.lower() == "stoptest":
			return self.TAG_AUTO_TEST_STOP, self.MSG_TAG_AUTO_TEST_STOP

		if s.lower() == "clearall":
			return self.TAG_CLEAR_ALL_TAGS, self.MSG_CLEAR_ALL_TAGS

		if s.lower() == "clearwifi":
			return self.TAG_CLEAR_WIFI_TAGS, self.MSG_CLEAR_WIFI_TAGS

		if s.lower() == "clearlora":
			return self.TAG_CLEAR_LORA_TAGS, self.MSG_CLEAR_LORA_TAGS

		if s.lower() == "kiss":
			return self.TAG_KISS, self.MSG_TAG_KISS
			
		if s.lower() == "status":
			return self.TAG_STATUS, self.MSG_TAG_STATUS

		print 'Unknown payload'
		
		return self.TAG_READ_ERROR, self.MSG_TAG_INVALID_PAYLOAD


	###################################################################################
	# NAME        : setFavorite
	# DESCRIPTION : Set the Favorite status
	################################################################################### 
	def setFavorite(self, state):
		self.waitingFavorite = state
	
	###################################################################################
	# NAME        : toggleFavorite
	# DESCRIPTION : Toggle favorite status
	################################################################################### 
	def toggleFavorite(self):
		self.waitingReference = False
	
		if self.waitingFavorite:
			self.waitingFavorite = False
			return self.TAG_OK, self.MSG_TAG_OK
		else:
			self.waitingFavorite = True
			return self.TAG_FAVORITE, self.MSG_TAG_FAVORITE

	###################################################################################
	# NAME        : isWaitingFavorite
	# DESCRIPTION : The system is waiting or not for a new tag to save as favorite
	# OUTPUT      : True  - Is waiting for another tag
	#             : False - 
	################################################################################### 
	def isWaitingFavorite(self):
		return self.waitingFavorite

	###################################################################################
	# NAME        : setReference
	# DESCRIPTION : Set the Reference status
	###################################################################################
	def setReference(self, state, ref):
		self.waitingReference = state
		self.whichReference = ref

	###################################################################################
	# NAME        : toggleReference
	# DESCRIPTION : Toggle reference status
	###################################################################################
	def toggleReference(self, ref):
		self.waitingFavorite = False

		if self.waitingReference:
			if self.whichReference == ref:
				self.waitingReference = False
				self.whichReference = None
				return self.TAG_OK, self.MSG_TAG_OK
			else:
				self.whichReference = ref
				return self.TAG_REFERENCE, self.MSG_TAG_REFERENCE + ref
		else:
			self.waitingReference = True
			self.whichReference = ref
			return self.TAG_REFERENCE, self.MSG_TAG_REFERENCE + ref

	###################################################################################
	# NAME        : isWaitingReference
	# DESCRIPTION : The system is waiting or not for a new tag to save as referenced
	# OUTPUT      : True  - Is waiting for another tag
	#             : False -
	###################################################################################
	def isWaitingReference(self):
		return self.waitingReference
	
	###################################################################################
	# NAME        : logTag
	# DESCRIPTION : Save the tag ID and information on log
	# INPUT       : tagId - tag uid
	################################################################################### 
	def logTag(self, tagId):
		
		if tagId == None or tagId == '':
			return self.TAG_READ_ERROR, self.MSG_TAG_OK
		
		dateStr = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
		
		if tagId in self.favorites:
			return self.toggleFavorite()
		else:
			if self.isWaitingFavorite():
				cardType = 'Favorite'
				self.setFavorite(False)
			else:
				if self.isWaitingReference():
					cardType = 'Reference' + self.whichReference
					self.setReference(False, None)
				else:
					cardType = 'Standard'
					# avoid duplicate last tag
					if self.lastTag == tagId:
						return self.TAG_OK, self.MSG_TAG_OK
		
		#save in csv format
		self.logFunction(",".join([tagId.upper(), dateStr, self.deviceId, cardType]))

		# try to post interaction
		#live_interaction.create(tagId)

		self.lastTag = tagId
		
		print("Tag saved: " + str(tagId))

		return self.TAG_OK_ADD, self.MSG_TAG_OK
	
