#!/usr/bin/python

from time import sleep
import threading
import os

from interface    import Interface
from tagsReader   import TagsReader
from update       import Update
from wifi         import Wifi
from logs         import Logs
from live         import Live


gui = []
up  = []
wifi= []
tr  = []
lg  = []
lv  = []
versionid = '0112'
versiondesc = ' - Production - Disconnect Wifi after boot. Not showing X. Add CLEAR tags'

###################################################################################
#                MAIN
###################################################################################
# NAME        : main
# DESCRIPTION : starts the system, handles received tags and graphical interface
################################################################################### 
def main():
	global gui, up, tr, lg, wifi, lv, versionid, versiondesc

	print('Version ' + versionid + versiondesc)


	gui = Interface()


	##read configuration and updates
	up = Update()
	lg = Logs(up.getBeamianId())
	tr = TagsReader(up.getId(), lg.writeLine)
	wifi = Wifi()
	lv = Live()

	gui.setId('ID: '+ up.getId())
	gui.setCompanyName(up.getCompanyName())

	gui.setLogo('data/logo.png')
	gui.setWifi(False)
	gui.setBluetooth(False)
	gui.setBeep(True)

	if firstRun():
		gui.setBeepFreq(4200)
		gui.makeBeep()
		sleep(0.1)
		gui.setBeepFreq(6000)
		gui.makeBeep()

		gui.setPopup("Checking new image")
		sleep(1)
		gui.updateScreenID()
		sleep(1)
		gui.updateScreen()
		firstRunSuccess()
		gui.stop()
		exitRestartApp()


	gui.setCounts(lg.numberLogsInFile())

	print('ID: ' + up.getBeamianId() + ' MAC: ' + up.getMac())
	gui.setPopup('ID:\n' + up.getBeamianId() + '\n\nMAC:\n' + up.getMac()[0:8] + '\n' + up.getMac()[9:17], 2)

	gui.setBeepFreq(6000)
	gui.makeBeep()
	sleep(0.1)
	gui.setBeepFreq(4200)
	gui.makeBeep()

	connectWifi(update,False)

	while True:

		sleep(0.5)
		code, msg = tr.waitAndSort()

		# error in read TAG
		if code == tr.TAG_READ_ERROR:
			continue

		gui.makeBeep()

		# Tag OK and increment
		if code == tr.TAG_OK_ADD:
			gui.setCounts(gui.getCounts() + 1)
			gui.updateScreen()
			if wifi.isConnected():
				connectWifi(kiss)
			continue

		# Tag OK; No increment
		if code == tr.TAG_OK:
			gui.updateScreen()
			if wifi.isConnected():
				connectWifi(kiss)
			continue

		# Favorite
		if code == tr.TAG_FAVORITE:
			gui.setPopup(msg, 0)
			continue

		# Reference
		if code == tr.TAG_REFERENCE:
			gui.setPopup(msg, 0)
			continue

		# Shutdown
		if code == tr.TAG_SHUTDOWN:
			break

		# Update
		if code == tr.TAG_UPDATE:
			connectWifi(update)
			continue

		# Upload
		if code == tr.TAG_UPLOAD:
			gui.makeBeep()
			connectWifi(upload)
			continue

		# Change ID and Name
		if code == tr.TAG_CHANGE_ID_NAME:
			changeIdName(msg)
			continue

		# Change BeamianId
		if code == tr.TAG_BEAMIAN_ID:
			changeBeamianId(msg)
			continue

		# Connect wifi
		if code == tr.TAG_WIFI_CONNECT:
			connectWifi(None, True)
			continue

		# Disconnect wifi
		if code == tr.TAG_WIFI_DISCONNECT:
			disconnectWifi()
			continue

		# Kiss (re-post last interaction)
		if code == tr.TAG_KISS:
			connectWifi(kiss)
			continue

		# Clear all tags
		if code == tr.TAG_CLEAR_ALL_TAGS:
			gui.makeBeep()
			gui.setPopup(msg, 2)
			lg.clearLogs()
			gui.setCounts(lg.numberLogsInFile())
			continue

		# Clear Wifi tags
		if code == tr.TAG_CLEAR_WIFI_TAGS:
			gui.makeBeep()
			gui.setPopup(msg, 2)
			lg.clearLogs()
			gui.setCounts(lg.numberLogsInFile())
			continue
		
		# Status (show status)
		if code == tr.TAG_STATUS:
			gui.setPopup(msg + '\n' + 'Version ' + versionid + '\nIp Address: ' + wifi.getIp(), 4)
			continue

		# Error in TAG
		if code == tr.TAG_ERROR:
			print(msg)
			gui.setPopup(msg, 2)

	gui.stop()
	exitShutdown()


###################################################################################
# NAME        : threadingConnectWifi
# DESCRIPTION : Tries to connect to the wifi. After that, can run a function
# INPUT       : nextFunction - Function to run after connection
################################################################################### 
def threadingConnectWifi(nextFunction):
	global wifi

	if wifi.isConnecting():
		if gui.getWifiStatus() == 2:
			gui.setWifi(True, 3)
		elif gui.getWifiStatus() == 3:
			gui.setWifi(True, 4)
		else:
			gui.setWifi(True, 2)

		threading.Timer(0.5, threadingConnectWifi, [nextFunction]).start()
		return False

	else:
		#No connection
		if not wifi.isConnected():
			print(       'Cannot connect to wireless network')
			gui.setPopup('Cannot connect to wireless network', 2)
			#gui.setWifi(True, -1) #too confusing for clients to see the "x"
			gui.setWifi(False)

			return False
		else:
			print(       'Ip Address: ' + wifi.getIp())
			gui.setPopup('Ip Address: ' + wifi.getIp(), 5)

			if nextFunction != None:
				nextFunction()

			return True

###################################################################################
# NAME        : connectWifi
# DESCRIPTION : Creates a tread to connect to the wifi
# INPUT       : nextFunction - Function to run after connection
#             : force        - Force a new connection
################################################################################### 	
def connectWifi(nextFunction=None, force=False):

	global wifi

	if force:
		wifi = Wifi(False)

	#Try connect
	if not wifi.isConnected():
		wifi.connect()

	threadingConnectWifi(nextFunction)

	return True

###################################################################################
# NAME        : disconnectWifi
# DESCRIPTION : Disconnect wifi interface
################################################################################### 
def disconnectWifi():

	global wifi, gui

	wifi.disconnect()
	gui.setWifi(False)

###################################################################################
# NAME        : update
# DESCRIPTION : Checks for firmware changes in server
################################################################################### 
def update():
	global gui

	print(       'Checking for updates')
	gui.setPopup('Checking for updates')
	sleep(1)
	version, data, msg = up.checkUpdates()
	gui.setPopup(msg, 2)

	if data !=  None:
		print msg
		v, m = up.updateData()
		gui.setPopup(m + '\n\n' + data, 2)
		sleep(1)

	if version !=  None:
		print msg
		v, m = up.updateFirmware()
		gui.setPopup(m + '\n\n' + version, 2)
		sleep(1)

		#valid image downloaded
		if v:
			gui.stop()
			exitUpdate()

	if version == None and data == None:
		disconnectWifi()
		sleep(1)

###################################################################################
# NAME        : upload
# DESCRIPTION : Send the log file to the FTP server
################################################################################### 			
def upload(delete=True):
	global gui, lg, up

	print(       'Uploading log file')
	gui.setPopup('Uploading log file', 0)

	try:
		up.sendConfig()
	except Exception, e:
		print(str(e))
	r, m = lg.sendLog(delete)

	print(m)
	gui.setCounts(lg.numberLogsInFile())
	gui.setPopup(m, 2)

	return r


###################################################################################
# NAME        : kiss
# DESCRIPTION : Send the last interaction to the API
###################################################################################
def kiss():
	global gui, lv

	print('Kissing API server')
	gui.setPopup('Uploading last interaction', 2)

	r, m = lv.sendLast()

	print(m)
	gui.setPopup(m, 2)

###################################################################################
# NAME        : changeBeamianId
# DESCRIPTION : Change the Beamian ID, including the hostname (in network).
#             : Needs to restart
################################################################################### 	
def changeBeamianId(msg):
	global gui, up
	up.newBeamianId(msg)

	print(       'New Beamian ID: \n' + msg)
	gui.setPopup('New Beamian ID: \n' + msg, 2)

	gui.stop()
	exitRestart()

###################################################################################
# NAME        : changeIdName
# DESCRIPTION : Change the displayed ID and Company Name. Need to restart app
# INPUT       : msg = [id,name]
################################################################################### 
def changeIdName(msg):
	global gui, up

	#default
	if msg == None or msg == '':
		up.setId(None)
		up.setCompanyName(None)
	else:
		msg = msg.split(',')

		if len(msg) >= 2:
			up.setId(msg[0])
			up.setCompanyName(msg[1])
		else:
			up.setId(msg[0])
			up.setCompanyName(None)

	print(       'New ID:\n' + up.getId() + '\n' + up.getCompanyName())
	gui.setPopup('New ID:\n' + up.getId() + '\n' + up.getCompanyName(), 2)
	sleep(2)

	gui.stop()
	exitRestartApp()

####################################
# Exit codes
####################################
def exitRestart():
	exit(1)

def exitShutdown():
	exit(2)

def exitUpdate():
	exit(3)

def exitRestartApp():
	exit(4)

####################################
# Check first time
####################################
def firstRun():
	#verify if file exist
	return not (os.path.isfile('.loaded.txt'))

def firstRunSuccess():
	#create loaded file
	return open('.loaded.txt', 'w+').close()

####################################
# System entry point
####################################	
if __name__ == "__main__":
	main()
